#!/bin/bash

# Ultra-minimal deployment script for DigitalOcean Ubuntu 22.04
# Run this on your server: bash deploy.sh

set -e  # Exit on error

# ========== CONFIGURATION - CHANGE THESE FOR MULTIPLE APPS ==========
APP_NAME="minimal-notes"                                    # Service name & directory
APP_PORT=3000                                               # Must be unique per app
APP_USER="app"                                               # System user (can share)
REPO_URL="${REPO_URL:-https://github.com/you/minimal-app.git}"  # Your git repo
SERVICE_DESCRIPTION="Minimal Notes App"                     # For systemd
DOMAIN="${DOMAIN:-}"                                        # Optional: your domain
# =====================================================================

echo "🚀 Starting deployment of $SERVICE_DESCRIPTION..."

# Check if running as root
if [ "$EUID" -eq 0 ]; then
   echo "✅ Running as root"
   IS_ROOT=true
else
   echo "📌 Running as user: $(whoami)"
   IS_ROOT=false
fi

# Function to prompt for domain if not set
get_domain() {
    if [ -z "$DOMAIN" ]; then
        read -p "Enter your domain (or press Enter to skip): " DOMAIN
    fi
    if [ -n "$DOMAIN" ]; then
        echo "📍 Using domain: $DOMAIN"
    else
        echo "⚠️  No domain provided - skipping nginx/SSL setup"
    fi
}

# Function to run commands as app user
run_as_app() {
    if [ "$IS_ROOT" = true ]; then
        su - $APP_USER -c "$1"
    else
        eval "$1"
    fi
}

# Function to check if port is already in use
check_port() {
    if command -v lsof >/dev/null 2>&1; then
        if lsof -Pi :$APP_PORT -sTCP:LISTEN -t >/dev/null 2>&1; then
            echo "⚠️  Warning: Port $APP_PORT is already in use!"
            echo "   Change APP_PORT at the top of this script to a different port"
            echo "   Common alternatives: 3001, 3002, 8080, 8081"
            exit 1
        fi
    fi
    echo "✅ Port $APP_PORT is available"
}

# 1. Initial Setup (if root)
if [ "$IS_ROOT" = true ]; then
    echo "📦 Setting up system..."

    # Create app user if doesn't exist
    if ! id "$APP_USER" &>/dev/null; then
        echo "Creating user: $APP_USER"
        adduser --gecos "" --disabled-password $APP_USER
        usermod -aG sudo $APP_USER
        echo "$APP_USER ALL=(ALL) NOPASSWD:ALL" >> /etc/sudoers
    fi

    # System updates
    apt-get update
    apt-get -y upgrade

    # Install dependencies
    apt-get -y install curl git nginx certbot python3-certbot-nginx ufw

    # Install Node.js LTS
    if ! command -v node &> /dev/null; then
        curl -fsSL https://deb.nodesource.com/setup_lts.x | bash -
        apt-get -y install nodejs
    fi

    # Configure firewall
    ufw --force enable
    ufw allow OpenSSH
    ufw allow 'Nginx Full'

    echo "✅ System setup complete"
fi

# 2. Clone/Update Repository
APP_DIR="/home/$APP_USER/$APP_NAME"

if [ ! -d "$APP_DIR" ]; then
    echo "📂 Cloning repository..."
    run_as_app "git clone $REPO_URL $APP_DIR"
else
    echo "📂 Updating repository..."
    run_as_app "cd $APP_DIR && git pull"
fi

# 3. Install Node Dependencies
echo "📦 Installing dependencies..."
run_as_app "cd $APP_DIR && npm install --production"

# Check if port is available
check_port

# 4. Initialize database
if [ ! -f "$APP_DIR/db.json" ]; then
    echo "🗄️ Initializing database..."
    run_as_app "echo '{\"notes\":[]}' > $APP_DIR/db.json"
fi

# 5. Setup Systemd Service
echo "⚙️ Setting up systemd service..."
sudo tee /etc/systemd/system/$APP_NAME.service > /dev/null <<EOF
[Unit]
Description=$SERVICE_DESCRIPTION
After=network.target

[Service]
Type=simple
User=$APP_USER
WorkingDirectory=$APP_DIR
Environment=NODE_ENV=production
Environment=PORT=$APP_PORT
ExecStart=/usr/bin/node server.js
Restart=always
RestartSec=3

[Install]
WantedBy=multi-user.target
EOF

# Reload and start service
sudo systemctl daemon-reload
sudo systemctl enable $APP_NAME
sudo systemctl restart $APP_NAME

echo "✅ Service started"

# 6. Setup Nginx (if domain provided)
if [ -n "$DOMAIN" ]; then
    echo "🌐 Domain provided: $DOMAIN"
elif [ "$IS_ROOT" = true ]; then
    get_domain
fi

if [ -n "$DOMAIN" ]; then

    echo "🌐 Setting up Nginx..."
    sudo tee /etc/nginx/sites-available/$DOMAIN > /dev/null <<EOF
server {
    listen 80;
    server_name $DOMAIN www.$DOMAIN;

    location / {
        proxy_pass http://127.0.0.1:$APP_PORT;
        proxy_http_version 1.1;
        proxy_set_header Host \$host;
        proxy_set_header X-Real-IP \$remote_addr;
        proxy_set_header X-Forwarded-For \$proxy_add_x_forwarded_for;
        proxy_set_header X-Forwarded-Proto \$scheme;

        # WebSocket support
        proxy_set_header Upgrade \$http_upgrade;
        proxy_set_header Connection "upgrade";
    }
}
EOF

    # Enable site
    sudo ln -sf /etc/nginx/sites-available/$DOMAIN /etc/nginx/sites-enabled/
    sudo nginx -t && sudo systemctl reload nginx

    echo "✅ Nginx configured"

    # 7. Setup SSL automatically
    echo "🔒 Setting up SSL certificate..."
    sudo certbot --nginx -d $DOMAIN -d www.$DOMAIN --non-interactive --agree-tos --register-unsafely-without-email
    echo "✅ SSL certificate installed"
fi

# 8. Show status
echo ""
echo "========================================="
echo "✨ Deployment Complete!"
echo "========================================="
echo ""
sudo systemctl status $APP_NAME --no-pager | head -10
echo ""
echo "📝 Useful commands:"
echo "  - View logs: journalctl -u $APP_NAME -f"
echo "  - Restart app: sudo systemctl restart $APP_NAME"
echo "  - Update app: cd $APP_DIR && git pull && npm i && sudo systemctl restart $APP_NAME"
echo ""

if [ -n "$DOMAIN" ]; then
    echo "🌐 Your app should be available at:"
    echo "   https://$DOMAIN"
else
    SERVER_IP=$(curl -s ifconfig.me)
    echo "🌐 Your app is available at:"
    echo "   http://$SERVER_IP:$APP_PORT"
    echo ""
    echo "To setup a domain, run:"
    echo "   DOMAIN=yourdomain.com bash deploy.sh"
fi

echo ""
echo "🎉 Happy coding!"